import { useStore, ToolMode } from '../store/useStore'
import { motion } from 'framer-motion'
import {
  Move3D,
  Hand,
  ZoomIn,
  Ruler,
  MessageSquare,
  RotateCcw,
  Maximize2,
  Grid3X3,
  Axis3D,
  RefreshCw,
  Camera,
  Download
} from 'lucide-react'

interface ToolButtonProps {
  icon: React.ReactNode
  label: string
  isActive?: boolean
  onClick: () => void
}

function ToolButton({ icon, label, isActive, onClick }: ToolButtonProps) {
  return (
    <button
      onClick={onClick}
      className={`p-3 rounded-xl transition-all flex items-center justify-center relative group
        ${isActive 
          ? 'bg-primary-500 text-white shadow-glow' 
          : 'bg-white hover:bg-surface-50 text-surface-600 hover:text-surface-800 shadow-soft'
        }`}
      title={label}
    >
      {icon}
      <span className="absolute left-full ml-2 px-2 py-1 bg-surface-800 text-white text-xs rounded opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-10">
        {label}
      </span>
    </button>
  )
}

export function ViewerControls() {
  const { toolMode, setToolMode, settings, updateSettings } = useStore()

  const tools: { mode: ToolMode; icon: React.ReactNode; label: string }[] = [
    { mode: 'orbit', icon: <Move3D size={20} />, label: 'Orbit' },
    { mode: 'pan', icon: <Hand size={20} />, label: 'Pan' },
    { mode: 'zoom', icon: <ZoomIn size={20} />, label: 'Zoom' },
    { mode: 'measure', icon: <Ruler size={20} />, label: 'Measure' },
    { mode: 'annotate', icon: <MessageSquare size={20} />, label: 'Annotate' },
  ]

  const handleScreenshot = () => {
    const canvas = document.querySelector('canvas')
    if (canvas) {
      const link = document.createElement('a')
      link.download = 'model-screenshot.png'
      link.href = canvas.toDataURL('image/png')
      link.click()
    }
  }

  const handleResetView = () => {
    // This would reset the camera - requires more integration with Three.js
    console.log('Reset view')
  }

  const handleFullscreen = () => {
    const viewer = document.querySelector('.flex-1.relative')
    if (viewer) {
      if (document.fullscreenElement) {
        document.exitFullscreen()
      } else {
        viewer.requestFullscreen()
      }
    }
  }

  return (
    <>
      {/* Left toolbar - Navigation tools */}
      <motion.div
        initial={{ opacity: 0, x: -20 }}
        animate={{ opacity: 1, x: 0 }}
        className="viewer-controls absolute left-4 top-1/2 -translate-y-1/2 flex flex-col gap-2"
      >
        {tools.map(({ mode, icon, label }) => (
          <ToolButton
            key={mode}
            icon={icon}
            label={label}
            isActive={toolMode === mode}
            onClick={() => setToolMode(mode)}
          />
        ))}
      </motion.div>

      {/* Bottom toolbar - View options */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className="viewer-controls absolute bottom-4 left-1/2 -translate-x-1/2 flex items-center gap-2 p-2 bg-white/90 backdrop-blur rounded-2xl shadow-medium"
      >
        <button
          onClick={() => updateSettings({ showGrid: !settings.showGrid })}
          className={`p-2.5 rounded-xl transition-colors ${
            settings.showGrid ? 'bg-primary-100 text-primary-600' : 'hover:bg-surface-100 text-surface-500'
          }`}
          title="Toggle Grid"
        >
          <Grid3X3 size={18} />
        </button>
        
        <button
          onClick={() => updateSettings({ showAxes: !settings.showAxes })}
          className={`p-2.5 rounded-xl transition-colors ${
            settings.showAxes ? 'bg-primary-100 text-primary-600' : 'hover:bg-surface-100 text-surface-500'
          }`}
          title="Toggle Axes"
        >
          <Axis3D size={18} />
        </button>

        <button
          onClick={() => updateSettings({ autoRotate: !settings.autoRotate })}
          className={`p-2.5 rounded-xl transition-colors ${
            settings.autoRotate ? 'bg-primary-100 text-primary-600' : 'hover:bg-surface-100 text-surface-500'
          }`}
          title="Auto Rotate"
        >
          <RefreshCw size={18} />
        </button>

        <div className="w-px h-6 bg-surface-200 mx-1" />

        <button
          onClick={handleResetView}
          className="p-2.5 rounded-xl hover:bg-surface-100 text-surface-500 transition-colors"
          title="Reset View"
        >
          <RotateCcw size={18} />
        </button>

        <button
          onClick={handleFullscreen}
          className="p-2.5 rounded-xl hover:bg-surface-100 text-surface-500 transition-colors"
          title="Fullscreen"
        >
          <Maximize2 size={18} />
        </button>

        <div className="w-px h-6 bg-surface-200 mx-1" />

        <button
          onClick={handleScreenshot}
          className="p-2.5 rounded-xl hover:bg-surface-100 text-surface-500 transition-colors"
          title="Take Screenshot"
        >
          <Camera size={18} />
        </button>

        <button
          className="p-2.5 rounded-xl hover:bg-surface-100 text-surface-500 transition-colors"
          title="Download Model"
        >
          <Download size={18} />
        </button>
      </motion.div>

      {/* Keyboard shortcuts hint */}
      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ delay: 1 }}
        className="absolute bottom-4 right-4 text-xs text-surface-400 hidden lg:block"
      >
        <span className="bg-surface-100 px-1.5 py-0.5 rounded mr-1">LMB</span> Rotate
        <span className="mx-2">|</span>
        <span className="bg-surface-100 px-1.5 py-0.5 rounded mr-1">RMB</span> Pan
        <span className="mx-2">|</span>
        <span className="bg-surface-100 px-1.5 py-0.5 rounded mr-1">Scroll</span> Zoom
      </motion.div>
    </>
  )
}


